﻿using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.GUI.controls;
using gov.va.med.vbecs.ExceptionManagement;
using gov.va.med.vbecs.GUI;
using gov.va.med.vbecs.GUI.WPF;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Windows.Forms;
using WPF.PresentationLayer.Resources;
using WPF.PresentationLayer.UC115.Events;
using WPF.PresentationLayer.UC115.Views;

namespace WPF.PresentationLayer.UC115.ViewModels
{
    /// <summary>
    /// SpecimenTabViewModel
    /// </summary>
    public class SpecimenTabViewModel : PendingTestViewModelBase
    {
        #region Properties
        public Patient Patient { get; private set; }
        /// <summary>
        /// Command figured when row checkbox is checked
        /// </summary>
        public DelegateCommand<SpecimenItemViewModel> ItemSelectedCommand { get; private set; }
        /// <summary>
        /// AcceptCommand
        /// </summary>
        public DelegateCommand AcceptCommand { get; set; }
        /// <summary>
        /// RejectCommand 
        /// </summary>
        public DelegateCommand RejectCommand { get; set; }
        /// <summary>
        /// Command fired when OrderReflex button is clicked
        /// </summary>
        public DelegateCommand OrderReflexCommand { get; private set; }
        /// <summary>
        /// Command fired when NewSIorTRC button is clicked
        /// </summary>
        public DelegateCommand NewSIorTRCommand { get; private set; }

        private ObservableCollection<SpecimenItemViewModel> _pendingSpecimenList;
        /// <summary>
        /// PendingSpecimenAboRhList - bound to DataGrid
        /// </summary>
        public ObservableCollection<SpecimenItemViewModel> PendingSpecimenList
        {
            get
            {
                return _pendingSpecimenList;
            }
            private set
            {
                _pendingSpecimenList = value;
                RaisePropertyChanged(() => PendingSpecimenList);
            }
        }
        #endregion

        /// <summary>
        /// SpecimenAboRhViewModel constructor
        /// </summary>
        /// <param name="vbecsBaseForm"></param>
        /// <param name="pendingSpecimenItemList"></param>
        public SpecimenTabViewModel(Patient patient, string specimenUid, VbecsBaseForm vbecsBaseForm, IEnumerable<SpecimenItemViewModel> pendingSpecimenItemList, StrRes.StringResourceItem helpFileTopic)
            : base(vbecsBaseForm, helpFileTopic)
        {
            Patient = patient;

            SetPendingSpecimenList(pendingSpecimenItemList);

            ItemSelectedCommand = new DelegateCommand<SpecimenItemViewModel>(OnItemSelectedCommand);
            AcceptCommand = new DelegateCommand(OnAcceptCommand);
            RejectCommand = new DelegateCommand(OnRejectCommand);
            OrderReflexCommand = new DelegateCommand(OnOrderReflexCommand);
            NewSIorTRCommand = new DelegateCommand(OnNewSIorTRCommand);
        }

        public void SetPendingSpecimenList(IEnumerable<SpecimenItemViewModel> pendingSpecimenItemList)
        {
            if (pendingSpecimenItemList == null)
            {
                PendingSpecimenList = null;
            }
            else
            {
                PendingSpecimenList = pendingSpecimenItemList.OrderByDescending(x => x.PendingSpecimenTestModel.TestedDateTime).ToObservableCollection();
            }
        }

        /// <summary>
        /// Fired when user clicks the Order Reflex button
        /// </summary>
        private void OnOrderReflexCommand()
        {
            if (VbecsBaseForm.IsCurrentUserAuthorizedForFunction(FunctionCode.StandardAccess))
            {

                // Defect 312073
                ArrayList orderedTestArrayList = new ArrayList();
                ArrayList TestGuids = new ArrayList();
            
                for (int idx = 0; idx < PendingSpecimenList.Count; idx++)
                {                              
                    if (!TestGuids.Contains(PendingSpecimenList[idx].PendingSpecimenTestModel.OrderedTest.OrderedTestGuid))
                    {
                        TestGuids.Add(PendingSpecimenList[idx].PendingSpecimenTestModel.OrderedTest.OrderedTestGuid);
                        orderedTestArrayList.Add(PendingSpecimenList[idx].PendingSpecimenTestModel.OrderedTest);
                    }
                }
                // End Defect 312073

                
                var orderedComponentArrayList = PendingSpecimenList.Select(x => x.PendingSpecimenTestModel as PendingSpecimenXMatchModel)
                                                                    .Where(w => w != null && w.OrderedComponentGuid.HasValue)
                                                                    .Select(s => new gov.va.med.vbecs.BOL.OrderedComponent(s.OrderedComponentGuid.Value))
                                                                    .ToArrayList();

                bool cancelForm = true;

                var frm = new FrmOrderSpecimenReflex(orderedTestArrayList,
                                                     orderedComponentArrayList,
                                                     ref cancelForm);
                if (cancelForm == false)
                {
                    frm.MdiParent = VbecsBaseForm.MdiParent;
                    frm.ShowFromLockedParent(VbecsBaseForm, false);
                    frm.WindowState = FormWindowState.Maximized;
                }
            }
            else
            {
                GuiMessenger.ShowMessageBox(StrRes.SysErrMsg.Common.InsufficientSecurityPrivileges("perform this function"));
            }
        }

        /// <summary>
        /// Fired when user clicks the New SI or TR button
        /// </summary>
        private void OnNewSIorTRCommand()
        {
            if (VbecsBaseForm.IsCurrentUserAuthorizedForFunction(FunctionCode.StandardAccess))
            {
                var frm = new FrmDefineTransfusionReqs(new Patient(Patient.PatientGuid)); // Creating a new patient to refresh SIs/TRs. This is not the best solution. It would be better to pass in a paraaeter to have the SIs/TRs refresh themselves, but we can't do that now.
                if (frm.DialogResult != DialogResult.Abort)
                {
                    frm.MdiParent = VbecsBaseForm.MdiParent;
                    frm.ShowFromLockedParent(VbecsBaseForm, true);
                    frm.WindowState = FormWindowState.Maximized;
                }
                else
                {
                    frm.Dispose();
                }
            }
            else
            {
                GuiMessenger.ShowMessageBox(StrRes.SysErrMsg.Common.InsufficientSecurityPrivileges("perform this function"));
            }
        }

        /// <summary>
        /// Reject the checked tests
        /// BR_115.05
        /// </summary>
        private void OnRejectCommand()
        {
            try
            {
                var selectedTestList = PendingSpecimenList.Where(x => x.IsSelected);

                if (selectedTestList.Any(x => string.IsNullOrWhiteSpace(x.ReviewComment)))
                {
                    GuiMessenger.ShowMessageBox(StrRes.ValidMsg.UC115.ReviewCommentRequired());
                    return;
                }

                var dlgResult = GuiMessenger.ShowMessageBox(StrRes.ConfMsg.UC115.RejectTest());
                if (dlgResult == DialogResult.Yes)
                {
                    var pendingTestModelList = selectedTestList.Select(s => s.PendingSpecimenTestModel).ToList();

                    // Store the review comment in the model before saving/rejecting
                    foreach (var testToReject in selectedTestList)
                    {
                        testToReject.PendingTestModel.ReviewComment = testToReject.ReviewComment;
                    }

                    // Make db call to reject tests                
                    PendingSpecimenTest.RejectTests(pendingTestModelList);

                    // Remove rejected tests from the DataGrid and full list
                    for (int idx = 0; idx < PendingSpecimenList.Count; idx++)
                    {
                        var pendingTest = PendingSpecimenList[idx];

                        if (pendingTest.IsSelected)
                        {
                            PendingSpecimenList.Remove(pendingTest);
                            idx--;
                        }
                    }

                    base.UncheckAll(PendingSpecimenList);

                    // Raise property changed on the list so the tab will check if it should be disabled
                    // if not tests exist anymore
                    RaisePropertyChanged(() => PendingSpecimenList);

                    // Fire event that tests were reviewed after we remove them from the list
                    Messenger.Default.Send(new PendingTestReviewedEvent(pendingTestModelList, false));
                }
            }
            catch (RowVersionException err)
            {
                ExceptionManager.Publish(err);
                GuiMessenger.ShowMessageBox(VbecsBaseForm, StrRes.SysErrMsg.Common.DataWasNotSavedBecauseOfRowversionViolation());
            }
        }

        /// <summary>
        /// Accept the checked tests
        /// BR_115.07
        /// </summary>
        private void OnAcceptCommand()
        {
            try
            {
                var selectedTestList = PendingSpecimenList.Where(x => x.IsSelected);
                var confirmationMessge = StrRes.ConfMsg.UC115.AcceptTest();

                // BR_56.01
                if (selectedTestList.Any(x => x.PendingTestModel is PendingSpecimenAboRhModel))
                {
                    confirmationMessge = StrRes.ConfMsg.UC056.SaveAboRh();
                }


                var dlgResult = GuiMessenger.ShowMessageBox(confirmationMessge);
                if (dlgResult == DialogResult.Yes)
                {
                    // Store the review comment and overrides in the model before saving/rejecting
                    foreach (var testToAccept in selectedTestList)
                    {
                        testToAccept.PendingTestModel.ReviewComment = testToAccept.ReviewComment;
                        testToAccept.PendingTestModel.TestValidation = testToAccept.TestValidation.Clone();

                        var testToAcceptXM = testToAccept as SpecimenXMatchItemViewModel;

                        // Save off XMatch specific properties
                        if (testToAcceptXM != null &&
                            testToAcceptXM.PendingSpecimenXMatchModel != null)
                        {
                            if (testToAcceptXM.PendingSpecimenXMatchModel.XMInterp != null &&
                                testToAcceptXM.SelectedXMatchInterp != null &&
                                testToAcceptXM.PendingSpecimenXMatchModel.XMInterp.TestResultId != testToAcceptXM.SelectedXMatchInterp.TestResultId)
                            {
                                var xmInterp = testToAcceptXM.PendingSpecimenXMatchModel.XMInterp;
                                xmInterp.TestResultId = testToAcceptXM.SelectedXMatchInterp.TestResultId;
                                xmInterp.TestResultText = testToAcceptXM.SelectedXMatchInterp.TestResultText;
                            }

                            if (testToAcceptXM.BloodUnitModel != null)
                            {
                                testToAcceptXM.PendingSpecimenXMatchModel.BloodUnitModel = testToAcceptXM.BloodUnitModel.Clone();
                                // Defect 284963
                                var orderedUnit = new OrderedUnit(testToAcceptXM.BloodUnitModel.BloodUnitGuid, Patient.PatientGuid);
                                testToAcceptXM.PendingSpecimenXMatchModel.OrderedUnitModel = new OrderedUnitModel(orderedUnit.OrderedUnitGuid, orderedUnit.SelectedDate, orderedUnit.RowVersion);
                            }
                        }
                    }

                    var pendingTestModelList = selectedTestList.Select(s => s.PendingSpecimenTestModel).ToList();

                    // Make db call to reject tests        
                    bool isWorkLoadsDefined = false;
                    PendingSpecimenTest.AcceptTests(LogonUser.LogonUserDivisionCode, pendingTestModelList, ref isWorkLoadsDefined);

                    if (!isWorkLoadsDefined)
                    {
                        GuiMessenger.ShowMessageBox(StrRes.InfoMsg.UC015.BR1502_NoWorkloadDefined());
                    }

                    // Remove rejected tests from the DataGrid and full list
                    for (int idx = 0; idx < PendingSpecimenList.Count; idx++)
                    {
                        var pendingTest = PendingSpecimenList[idx];

                        if (pendingTest.IsSelected)
                        {
                            PendingSpecimenList.Remove(pendingTest);
                            idx--;
                        }
                    }

                    base.UncheckAll(PendingSpecimenList);

                    // Raise property changed on the list so the tab will check if it should be disabled
                    // if no tests exist anymore
                    RaisePropertyChanged(() => PendingSpecimenList);

                    PromptToPrintXMLabelsIfNeeded(pendingTestModelList.Select(s => s as PendingSpecimenXMatchModel).Where(x => x != null).ToList());

                    // Fire event that tests were reviewed after we remove them from the list
                    Messenger.Default.Send(new PendingTestReviewedEvent(pendingTestModelList, false));
                }
            }
            catch (RowVersionException err)
            {
                ExceptionManager.Publish(err);
                GuiMessenger.ShowMessageBox(VbecsBaseForm, StrRes.SysErrMsg.Common.DataWasNotSavedBecauseOfRowversionViolation());
            }
        }

        /// <summary>
        /// Print XM labels if user is accepting a comptable XM
        /// </summary>
        /// <param name="pendingTestModelList"></param>
        private void PromptToPrintXMLabelsIfNeeded(List<PendingSpecimenXMatchModel> pendingXMTestModelList)
        {
            if (pendingXMTestModelList != null)
            {
                var printBloodUnitGuids = pendingXMTestModelList.Where(x => x.BloodUnitModel != null &&
                                                            x.XMInterp != null &&
                                                           (x.XMInterp.TestResultId == PendingTestHelper.COMPATIBLE_SAFE_TO_TRANSFUSE_ID ||
                                                            x.XMInterp.TestResultId == PendingTestHelper.INCOMPATIBLE_ONLY_MEDICAL_DIRECTOR_ID))
                                                           .Select(s => s.BloodUnitModel.BloodUnitGuid);

                if (printBloodUnitGuids.Any())
                {
                    DialogResult result = GuiMessenger.ShowMessageBox(StrRes.ConfMsg.UC040.PrintBTRFandCautionTag());

                    if (result == DialogResult.Yes)
                    {
                        FrmPrintCautionTagByPatient frm = new FrmPrintCautionTagByPatient(Patient, printBloodUnitGuids.ToArrayList());
                        if (frm.IsValid)
                        {
                            frm.MdiParent = VbecsBaseForm.MdiParent;
                            frm.Show();
                        }
                    }
                }
            } 
        }

        /// <summary>
        /// Handles checking/unchecking a given test
        /// </summary>
        /// <param name="selectableTest"></param>
        public void OnItemSelectedCommand(ISelectableTestViewModel selectableTest)
        {
            base.ToggleCheckBoxesEnabledState(selectableTest, PendingSpecimenList);

            var selectedXM = selectableTest as SpecimenXMatchItemViewModel;

            // BR_115.33 - No Product code exists so we need to prompt the user to select one
            if (selectedXM != null &&
                selectedXM.BloodUnitModel == null)
            {
                PromptForBloodUnit(selectedXM);

                if (selectedXM.BloodUnitModel == null)
                {
                    UncheckSelectedTest(selectableTest);
                }
            }

            var selectedSpecimen = selectableTest as SpecimenItemViewModel;

            if (selectedSpecimen != null &&
                selectableTest != null &&
                selectableTest.IsSelected)
            {
                selectedSpecimen.PromptUserWithWarnings(VbecsBaseForm);

                selectedSpecimen.PromptForExceptionOverrides(VbecsBaseForm);

                if (!selectedSpecimen.AreAllExceptionOverridesProcessed())
                {
                    UncheckSelectedTest(selectableTest);
                }
            }
        }

        /// <summary>
        /// Uncheck the given test when the UI threads available again via Dispatcher.BeginInvoke 
        /// </summary>
        /// <param name="selectableTest"></param>
        private void UncheckSelectedTest(ISelectableTestViewModel selectableTest)
        {
            selectableTest.IsSelected = false;
            // No product code was selected so we will uncheck the user's selection
            System.Windows.Application.Current.Dispatcher.BeginInvoke(new System.Action(() =>
            {
                selectableTest.IsSelected = false;
                base.ToggleCheckBoxesEnabledState(selectableTest, PendingSpecimenList);
            }));
        }

        /// <summary>
        /// Check for unsaved changes
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool IsDirty()
        {
            foreach (var testViewModel in PendingSpecimenList)
            {
                if (testViewModel.IsDirty())
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// Prompt the user to enter a product code
        /// </summary>
        /// <param name="selectedTest"></param>
        private void PromptForBloodUnit(SpecimenXMatchItemViewModel selectedTest)
        {
            if (selectedTest != null)
            {
                var originalTestValidationStatus = TestValidationStatus.Unknown;
                if (selectedTest.TestValidation != null)
                {
                    originalTestValidationStatus = selectedTest.TestValidation.TestValidationStatus;
                }

                //Defect 282082
                if (selectedTest.PendingSpecimenXMatchModel != null)
                {
                    var dlg = ViewConstructor.CreateFormAndView<BloodUnitProductCodeView, FrmWPFContainer>();
                    var dlgViewModel = new BloodUnitProductCodeViewModel(selectedTest.PendingSpecimenXMatchModel.BloodUnitId, selectedTest.PendingSpecimenXMatchModel.SpecimenUid, selectedTest.BloodUnitModel, dlg);
                    dlgViewModel.LogoHeaderText = "Select Product Code";
                    dlg.WPFUserControl.DataContext = dlgViewModel;
                    var dlgResult = dlg.ShowDialogFromLockedParent(VbecsBaseForm, false);

                    if (dlgResult == DialogResult.OK &&
                        dlgViewModel.SelectedBloodUnit != null)
                    {
                        selectedTest.BloodUnitModel = dlgViewModel.SelectedBloodUnit.Clone();

                        if (selectedTest.TestValidation == null ||
                            originalTestValidationStatus != selectedTest.TestValidation.TestValidationStatus)
                        {
                            // Test Validation changed so uncheck all other tests
                            // and select this test
                            base.UncheckAll(PendingSpecimenList);

                            System.Windows.Application.Current.Dispatcher.BeginInvoke(new System.Action(() =>
                            {
                                selectedTest.IsSelected = true;
                                OnItemSelectedCommand(selectedTest);
                            }));
                        }
                    }
                }
            }
        }
    }
}
